// Copyright 2025 Specter Ops, Inc.
//
// Licensed under the Apache License, Version 2.0
// http://www.apache.org/licenses/LICENSE-2.0
//
// SPDX-License-Identifier: Apache-2.0

import userEvent from '@testing-library/user-event';
import { AssetGroupTag } from 'js-client-library';
import { UseQueryResult } from 'react-query';
import { render, screen } from '../../../test-utils';
import SummaryList from './SummaryList';

vi.mock('../../../components/AppIcon/Icons/DownArrow', () => ({
    default: () => <div data-testid='privilege-zones_summary-list_down-arrow' />,
}));

vi.mock('./SummaryCard', () => ({
    default: ({ title }: { title: string }) => <div data-testid='privilege-zones_summary-list_card'>{title}</div>,
}));

vi.mock('../Details/utils', () => ({
    itemSkeletons: [
        () => <li key='skeleton-1' data-testid='privilege-zones_zone-list_loading-skeleton' />,
        () => <li key='skeleton-2' data-testid='privilege-zones_zone-list_loading-skeleton' />,
        () => <li key='skeleton-3' data-testid='privilege-zones_zone-list_loading-skeleton' />,
    ],
}));

const mockData: AssetGroupTag[] = [
    {
        id: 1,
        name: 'Mock Zone 1',
        kind_id: 10,
        type: 1,
        position: 1,
        glyph: null,
        requireCertify: false,
        description: 'A mock tag for testing',
        created_at: '2024-01-01T00:00:00Z',
        updated_at: '2024-01-02T00:00:00Z',
        deleted_at: null,
        created_by: 'user',
        updated_by: 'user',
        deleted_by: null,
        counts: {
            selectors: 5,
            members: 10,
        },
        analysis_enabled: false,
    },
    {
        id: 2,
        name: 'Mock Zone 2',
        kind_id: 20,
        type: 1,
        position: 2,
        glyph: null,
        requireCertify: true,
        description: 'Another mock tag',
        created_at: '2024-01-03T00:00:00Z',
        updated_at: '2024-01-04T00:00:00Z',
        deleted_at: null,
        created_by: 'user',
        updated_by: 'user',
        deleted_by: null,
        counts: {
            selectors: 3,
            members: 6,
        },
        analysis_enabled: false,
    },
];

const expectedCount = mockData.filter((item) => item.type === 1).length;

describe('SummaryList', () => {
    it('shows skeletons when loading', () => {
        const query = {
            isLoading: true,
            isError: false,
        } as unknown as UseQueryResult<AssetGroupTag[]>;

        render(<SummaryList title='Zones' selected='' listQuery={query} onSelect={() => {}} />);

        expect(screen.getAllByTestId('privilege-zones_zones-list_loading-skeleton')).toHaveLength(3);
    });

    it('shows an error message when query fails', async () => {
        const query = {
            isLoading: false,
            isError: true,
        } as unknown as UseQueryResult<AssetGroupTag[]>;

        render(<SummaryList title='Zones' selected='' listQuery={query} onSelect={() => {}} />);

        expect(await screen.findByText('There was an error fetching this data')).toBeInTheDocument();
    });

    it('renders sorted list items by name descending', async () => {
        const query = {
            isSuccess: true,
            data: mockData,
        } as unknown as UseQueryResult<AssetGroupTag[]>;

        render(<SummaryList title='Zones' selected='' listQuery={query} onSelect={() => {}} />);

        const cards = await screen.findAllByTestId('privilege-zones_summary-list_card');
        expect(cards[0]).toHaveTextContent('Mock Zone 1');
        expect(cards[1]).toHaveTextContent('Mock Zone 2');
    });

    it('renders a down arrow only for items of type 1', async () => {
        const query = {
            isSuccess: true,
            data: mockData,
        } as unknown as UseQueryResult<AssetGroupTag[]>;

        render(<SummaryList title='Zones' selected='' listQuery={query} onSelect={() => {}} />);

        const arrows = await screen.findAllByTestId('privilege-zones_summary-list_down-arrow');
        expect(arrows).toHaveLength(expectedCount);
    });

    it('calls onSelect when a card is clicked', async () => {
        const onSelect = vi.fn();

        const query = {
            isSuccess: true,
            data: [mockData[0]],
        } as unknown as UseQueryResult<AssetGroupTag[]>;

        render(<SummaryList title='Zones' selected='' listQuery={query} onSelect={onSelect} />);

        await userEvent.click(await screen.findByTestId('privilege-zones_summary-list_card'));
        expect(onSelect).toHaveBeenCalledWith(mockData[0].id);
    });
});
