// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (c) 2016-2025 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 */
#include <linux/acpi.h>
#include <linux/clk.h>
#include <linux/device.h>
#include <linux/gpio.h>
#include <linux/i2c.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/gpio/consumer.h>
#include <linux/platform_device.h>
#include <linux/regulator/consumer.h>
#include <linux/regulator/driver.h>
#include <linux/regulator/fixed.h>
#include <linux/regulator/gpio-regulator.h>
#include <linux/regulator/machine.h>
#include <linux/list.h>
#include <linux/platform_device.h>
#include <linux/clkdev.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <media/ipu-acpi-pdata.h>
#include <media/ipu-acpi.h>

#if IS_ENABLED(CONFIG_VIDEO_ISX031)
#include <media/i2c/isx031.h>
#endif

#include "ipu7.h"
#include "ipu7-isys.h"

static LIST_HEAD(devices);

static struct ipu_camera_module_data *add_device_to_list(
	struct list_head *devices)
{
	struct ipu_camera_module_data *cam_device;

	cam_device = kzalloc(sizeof(*cam_device), GFP_KERNEL);
	if (!cam_device)
		return NULL;

	list_add(&cam_device->list, devices);
	return cam_device;
}

static const struct ipu_acpi_devices supported_devices[] = {
/*
 *	{ "ACPI ID", sensor_name, get_sensor_pdata, NULL, 0, TYPE, serdes_name,
 *		sensor_physical_addr, link_freq(mbps) },	// Custom HID
 */

#if IS_ENABLED(CONFIG_VIDEO_MAX9X)
#if IS_ENABLED(CONFIG_VIDEO_ISX031)
	{ "INTC031M", ISX031_NAME, get_sensor_pdata, NULL, 0, TYPE_SERDES, "max9x",
		ISX031_I2C_ADDRESS, 1600 },	// D3 ISX031 HID
#endif
#endif
};

static int get_table_index(const char *acpi_name)
{
	unsigned int i;

	for (i = 0; i < ARRAY_SIZE(supported_devices); i++) {
		if (!strncmp(supported_devices[i].hid_name, acpi_name,
			     strlen(supported_devices[i].hid_name)))
			return i;
	}

	return -ENODEV;
}

/* List of ACPI devices what we can handle */
/* Must match with HID in BIOS option. Add new sensor if required */
static const struct acpi_device_id ipu_acpi_match[] = {
/*
 *	{ "AR0234A", 0 },	// Custom HID
 */
#if IS_ENABLED(CONFIG_VIDEO_ISX031)
	{ "INTC1031", 0 },	// ISX031 HID
	{ "INTC031M", 0 },	// D3CMC68N-115-084 ISX031 HID
#endif
	{},
};

static int ipu_acpi_get_pdata(struct device *dev, int index)
{
	struct ipu_camera_module_data *camdata;
	int rval;

	if (index < 0) {
		pr_err("Device is not in supported devices list\n");
		return -ENODEV;
	}

	camdata = add_device_to_list(&devices);
	if (!camdata)
		return -ENOMEM;

	pr_info("IPU ACPI: Getting BIOS data for %s (%s)",
		supported_devices[index].real_driver, dev_name(dev));

	rval = supported_devices[index].get_platform_data(
		dev, camdata,
		supported_devices[index].priv_data,
		supported_devices[index].priv_size,
		supported_devices[index].connect,
		supported_devices[index].real_driver,
		supported_devices[index].serdes_name,
		supported_devices[index].hid_name,
		supported_devices[index].sensor_physical_addr,
		supported_devices[index].link_freq);

	if (rval)
		return -EPROBE_DEFER;

	return 0;
}

/*
 * different acpi devices may have same HID, so acpi_dev_get_first_match_dev
 * will always match device to simple fwnode.
 */
static int ipu_acpi_test(struct device *dev, void *priv)
{
	struct acpi_device *adev = NULL;
	int rval;
	int acpi_idx = get_table_index(dev_name(dev));

	if (acpi_idx < 0)
		return 0;
	else
		dev_info(dev, "IPU6 ACPI: ACPI device %s\n", dev_name(dev));

	const char *target_hid = supported_devices[acpi_idx].hid_name;

	if (!ACPI_COMPANION(dev)) {
		while ((adev = acpi_dev_get_next_match_dev(adev, target_hid,
							   NULL, -1))) {
			if (adev->flags.reserved == 0) {
				adev->flags.reserved = 1;
				break;
			}
			acpi_dev_put(adev);
		}

		if (!adev) {
			dev_dbg(dev, "No ACPI device found for %s\n", target_hid);
			return 0;
		} else {
			set_primary_fwnode(dev, &adev->fwnode);
			dev_dbg(dev, "Assigned fwnode to %s\n", dev_name(dev));
		}
	}

	if (ACPI_COMPANION(dev) != adev) {
		dev_err(dev, "Failed to set ACPI companion for %s\n",
			dev_name(dev));
		acpi_dev_put(adev);
		return 0;
	}

	acpi_dev_put(adev);

	rval = ipu_acpi_get_pdata(dev, acpi_idx);
	if (rval) {
		pr_err("IPU6 ACPI: Failed to process ACPI data");
		return rval;
	}

	return 0; /* Continue iteration */
}

/* Try to get all IPU related devices mentioned in BIOS and all related information
 * return a new generated existing pdata
 */

int ipu_get_acpi_devices(void **spdata)
{
	int rval;
	struct ipu7_isys_subdev_pdata *ptr = NULL;

	rval = acpi_bus_for_each_dev(ipu_acpi_test, NULL);
	if (rval < 0)
		return rval;

	ptr = get_acpi_subdev_pdata();
	if (ptr && *ptr->subdevs)
		*spdata = ptr;

	return 0;
}
EXPORT_SYMBOL(ipu_get_acpi_devices);

static int __init ipu_acpi_init(void)
{
	return 0;
}

static void __exit ipu_acpi_exit(void)
{
}

module_init(ipu_acpi_init);
module_exit(ipu_acpi_exit);

MODULE_AUTHOR("Samu Onkalo <samu.onkalo@intel.com>");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("IPU ACPI support");
